/* Copyright (C) 1998, Mike Butler, mgb@mitre.org
 *    
 * This file is part of picptk, a PIC programmer for Linux
 *
 * Picptk is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * Picptk is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with picptk; see the file COPYING.  If not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 * 
 * $Id: UtRealTime.h,v 1.2 1998/12/31 01:40:33 mgb Exp $
 */
/* Linux kernel manipulations to use soft real-time programming
 */


#ifndef RealTime_h
#define RealTime_h

#include <time.h>		// for uint
#include <sched.h>		// for sched_setscheduler()

#include <errno.h>		// for errno

class RealTime {
private:
  struct sched_param cOldParam;
  int cOldPolicy;
  enum { kDefaultPriority = 50 };
public:
  RealTime(int priority = kDefaultPriority, int policy = SCHED_FIFO) { 
    struct sched_param param;
    
    const char *death = "Can't change kernel scheduler or policy";
    if((cOldPolicy = sched_getscheduler(0)) < 0) 
      throw(death);
    if(sched_getparam(0, &cOldParam)) 
      throw(death);
    param = cOldParam;
    param.sched_priority = priority;
    if(sched_setscheduler(0, policy, &param))       
      throw(death);
  }
  ~RealTime() {
    if(sched_setscheduler(0, cOldPolicy, &cOldParam)) 
      throw("Can't reset kernel scheduler or policy");
  }
  static void DelayUs(unsigned long usec) {
    timespec ts;
    ts.tv_sec = usec / 1000000;
    ts.tv_nsec = (usec % 1000000) * 1000L;
    nanosleep(&ts, 0);
  }
};

#endif // RealTime_h
