/* Copyright (C) 1998, Mike Butler, mgb@mitre.org
 *    
 * This file is part of picptk, a PIC programmer for Linux
 *
 * Picptk is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * Picptk is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with picptk; see the file COPYING.  If not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * $Id: UtEnvLite.h,v 1.2 1998/12/31 01:40:33 mgb Exp $
 *
 * This is an environemnt manipulation class...
 * It provides a simple, common way for programs to manipulate
 * configuration options.  This is a simplified version, the original
 * (and considerably larger) UtEnvironemnt implements a more
 * complete environment manipulation package
 */
#ifndef _Environment_h_
#define _Environment_h_

#include <fstream.h>
#include <stdio.h>
#include <stdlib.h>

class Environment {
private:
  struct Attribute {
    char *cName;
    char *cValue;
    bool cDefaulted;
    Attribute *cNext;
  };

  Attribute *cHead;
  const char *cFile;

  // Look up an attribute by name, return pointer to it...
  Attribute *Find(const char *name) {
    Attribute *pa;
    for(pa = cHead; pa; pa = pa->cNext)
      if(!strcmp(name, pa->cName)) return(pa);
    return(0);
  }

public:
  // Construct environment and set default prefs file.
  Environment(const char *file = 0) { 
    cHead = 0; 
    cFile = 0; 
    if(file) DefaultFile(file);
  }

  ~Environment() { 
    delete [] cFile;
    Clear();
  }

  // Set the default file for loads and saves...
  void DefaultFile(const char *file) {
    if(cFile) delete [] cFile;
    char *tmp = new char[strlen(file)+1];
    strcpy(tmp, file);
    cFile = tmp;
  }

  // Clear all preferences...
  void Clear() {
    while(cHead) Unset(cHead->cName);
  }

  // delete and attribute...
  bool Unset(const char *name) {
    Attribute *curr, *prev = 0;
    for(curr = cHead; curr; prev = curr, curr = curr->cNext) {
      if(!strcmp(name, curr->cName)) {
	// Found it, splice around it and then delete it...
	if(prev) prev->cNext = curr->cNext;
	else cHead = curr->cNext;
	delete [] curr->cValue;
	delete [] curr->cName;
	delete curr;
	return(true);
      }
    }
    // Not found?
    return(false);
  }
  // Set an attribute, defining it if needed...
  // Deletes attribute if value is null...
  const char *Set(const char *name, const char *val, bool def = false) {
    if(!val) {
      Unset(name);
      return("");
    }

    Attribute *pa = Find(name);
    if(!pa) {
      // Not there, splice in a new node...
      pa = new Attribute;
      pa->cNext = cHead;
      cHead = pa;

      pa->cName = new char[strlen(name)+1];
      strcpy(pa->cName, name);
      pa->cValue = new char[strlen(val)+1];
      strcpy(pa->cValue, val);
      pa->cDefaulted = def;		// a "soft" setting...
    } else {
      // Change value on this attribute (unless this is a default)
      if(!def) {
	delete [] pa->cValue;
	pa->cValue = new char[strlen(val)+1];
	strcpy(pa->cValue, val);
	pa->cDefaulted = def;
      }
    }
    // Note that this may differ from the value passed!
    return(pa->cValue);
  }

  /* Look up a value, define a default value if non is found.
   * If default is used, a new attribute is set to this value
   * and that setting appears as a comment if the environment is saved.
   * (This exposed the environment variables to the user w/o hard coding
   * them all.)
   * Callers should can pass a default value...  That'll get the
   * variable defined in the environmnet as a default...
   */
  const char *Get(const char *name, const char *def = 0) {
    Attribute *pa = Find(name);
    if(!pa) {
      if(def) 
	return(Set(name, def, true));
      return(0);
    }
    return(pa->cValue);
  }

  // Read environment from a file...
  // Returns either file name on success, or null on failure
  const char *Load(const char *file = 0) {
    if(!file) file = cFile;
    if(!file) return(0);
    ifstream is(file);
    if(!is) return(0);


    const int kMaxLine = 100;
    char line[kMaxLine], *a, *v;

    while(is) {
      is.getline(line, kMaxLine);
      if(!is) continue;		// What to do here?
      for(a = line; *a == ' '; a++) ;
      if(*a == ';') continue;	// Comment line...

      // With what's left, look for attribute=value thing...
      for(v = a; *v ; v++) 
	if(*v == '=') break;	// Assignment
      if(!*v) continue;		// No equals?  Ignore line...
      *v++ = 0;			// break into attr/value
      while(*v == ' ') v++;
      Set(a, v);		// Maybe make new attrribute...
    }
    return(file);
  }

  // A naked Save uses the default file name...
  const char *Save(const char *file = 0) {
    if(!file) file = cFile;
    if(!file) return(0);
    ofstream os(file);
    if(!os) return(0);

    os << "; Init file for picp, the generic PIC programmer driver." << endl;
    os << "; Lines whose first non-blank is ';' are comments." << endl;
    os << "; Blank lines (only white space) are also ignored." << endl;
    os << "; Attribute assignments should be of the form: " << endl;
    os << ";     attr=value" << endl;
    os << "; Leading space is not significant, but trailing space is." << endl;
    os << "; " << endl;
    os << "; Default attribute values are included as comments " << endl;
    os << "; (So when they change this file won't override them)" << endl;
    os << endl;

    for(Attribute *pa = cHead; pa; pa = pa->cNext) {
      if(pa->cDefaulted) os << ";";
      os << pa->cName << "=" << pa->cValue << endl;
    }
    return(file);
  }
};

#endif // _Environemnt_h_
