/* Copyright (C) 1998, Mike Butler, mgb@mitre.org
 *    
 * This file is part of picptk, a PIC programmer for Linux
 *
 * Picptk is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * Picptk is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with picptk; see the file COPYING.  If not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * $Id: UiItk.C,v 1.5 1998/12/31 01:40:33 mgb Exp mgb $
 *
 * incrTk interface to PIC programmer core...
 *
 * The interface is just glommed onto the core C routines.  This
 * allows one to develop the GUI independent of the core.  (I had
 * anticipated profiding a command line interface, but don't yet have
 * a the need...)
 *
 */

#include <UiItk.h>

#ifdef HAVE_LIBITK3_0
#include <itk.h>
#endif

#include <strstream.h>		// for ostrstream


/* Gui dispatch table for TCL commands...
 * We register only one command in the itk interpreter, all
 * functions are subcommands off that one.
 * The table holds the
 *      1) subcommand name
 *      2) a pointer to the function that accomplishes that command
 *      3) the number of arguments expected
 */
Gui::HandlerBinding_t Gui::cHandlers[] = {
  { "load",   Gui::DoLoad,      1,      "File Load"   },
  { "chip",   Gui::DoChip,      1,      0             },
  { "port",   Gui::DoPort,      1,      0             },

  { "read",   Gui::DoRead,      0,      "Pic Read"    },
  { "erase",  Gui::DoErase,     0,      "Pic Erase"   },
  { "burn",   Gui::DoBurn,      0,      "Pic Burn"    },
  { "dump",   Gui::DoDump,      0,      "Mem Show"    },
  { "verify", Gui::DoVerify,    0,      "Pic Verify"  },
   
  { "chips",  Gui::DoChips,     0,      0             }, // not user visible
  { "pins",   Gui::DoPins,      0,      0             }, // not user visible
  { "info",   Gui::DoInfo,      0,      0             }, // not user visible
  { "version",Gui::DoVersion,   0,      0             }, // not user visible
   
  { "set",    Gui::DoSet,       2,      0             },
  { "get",    Gui::DoGet,       1,      0             },
  { "loadp",  Gui::DoLoadPrefs, 1,      "Pref Load"   },
  { "savep",  Gui::DoSavePrefs, 1,      "Pref Save"   },
};



/* Handle a specific subcommand command on this object...
 * This is the subcommand dispatcher.  It checks
 * command name validity, number of arguments, and invokes the
 * appropriate Do'er
 */
int
Gui::TclExec(ClientData clientData,
	     Tcl_Interp *interp,
	     int argc, char *argv[]) {

  if(argc < 2) {
    Tcl_AppendResult(interp, "Wrong number of args to ", argv[0], 0);
    return(TCL_ERROR);
  }

  // TCL guarantees argv[argc] is zero, so this is safe, right?
  char *obj = argv[0];
  char *cmd = argv[1];
  Gui *me = (Gui *)clientData;

  // Search through table of known commands and dispatch...
  for(uint i = 0; i < sizeof(cHandlers)/sizeof(cHandlers[0]); i++) {
    if(!strcmp(cmd, cHandlers[i].cName)) {
      // Assume bad until proven otherwise...
      if(argc-2 != cHandlers[i].cArgs) {
	Tcl_AppendResult(interp, "Wrong number of args to ", obj, " ", cmd, 0);
	return(TCL_ERROR);
      }
      // Looks good, do exec and handle errors...
      // Called routine returns zero for success, error count
      // otherwise.  Messages are pretty stock...
      try {
	int errors = cHandlers[i].cProc(me, interp, argc, argv);
	if(errors) {
	  char tmp[10];
	  sprintf(tmp, "%d", errors);
	  Tcl_AppendResult(interp, "Failure - ", tmp, " errors\n", 0);
	  return(TCL_ERROR);
	}
	if(cHandlers[i].cPrintName) 
	  Tcl_AppendResult(interp, "Success - ", 
			   cHandlers[i].cPrintName, " OK\n", 0);
	return(TCL_OK);
      } catch (const char *e) {
	Tcl_AppendResult(interp, "Failure - ", e, "\n", 0);
	return(TCL_ERROR);
      }
    }
  }

  // Got here?  Command not known...
  Tcl_AppendResult(interp, "Unknown command '", obj, " ", cmd, "'\n", 0);
  return(TCL_ERROR);
}

//
/* Load PIC RAM image from file...
 *
 */
int Gui::DoLoad(Gui *me, Tcl_Interp *interp, int argc, char *argv[]) {
  const char *file = argv[2];
  char tmp[20];
  int bytes = me->cPic.LoadRam(file);
  sprintf(tmp, "%d", bytes);
  Tcl_AppendResult(interp, "Loaded ", tmp, " bytes from '", file, "'\n", 0);
  return(0);
}

/* This is a generic way for a Gui to to save and restore
 * preferences...
 * The values are read automatically at startup by the Pic class, but
 * They are never automatically written...
 *
 * Please do not use Get and Set to subvert the select, load, port,
 * and other commands...  
 * You'll violate encapsulation and likely break something!
 */
/* Set a preference variable... */
int Gui::DoSet(Gui *me, Tcl_Interp *interp, int argc, char *argv[]) {
  Tcl_AppendResult(interp, me->cPic.SetPref(argv[2], argv[3]), 0);
  return(0);
}
/* Set a preference variable... */
int Gui::DoGet(Gui *me, Tcl_Interp *interp, int argc, char *argv[]) {
  const char *val = me->cPic.GetPref(argv[2]);
  if(val) Tcl_AppendResult(interp, val, 0);
  return(0);
}
/* Save preference file... */
int Gui::DoSavePrefs(Gui *me, Tcl_Interp *interp, int argc, char *argv[]) {
  char *arg = argv[2];
  if(!arg[0]) return(!me->cPic.SavePrefs());
  else return(!me->cPic.SavePrefs(arg));
}

int Gui::DoLoadPrefs(Gui *me, Tcl_Interp *interp, int argc, char *argv[]) {
  char *arg = argv[2];
  if(!arg[0]) return(!me->cPic.LoadPrefs());
  else return(!me->cPic.LoadPrefs(arg));
}


/* Select a specific PIC device type...
 *
 * Information about PIC devices is kept in a table in the C code.
 * To add support for a new PIC type, edit the table.  (You should not
 * have to change any code unless the device has some peculiar
 * programming requirements.
 */
int
Gui::DoChip(Gui *me, Tcl_Interp *interp, int argc, char *argv[]) {
  const char *arg = argv[2];		// checked previously
  if(!arg[0]) {
    // Query for default chip query...
    Tcl_AppendResult(interp, me->cPic.SetChip(), 0);
  } else {
    arg = me->cPic.SetChip(arg);
    Tcl_AppendResult(interp, "PIC type set to ", arg, "\n", 0);
  }
  return(0);
}

/* Select a specific for the programmer port...
 *
 * Programmer specifics are well isolated, so this programmer can
 * support both serial and parallel programmers.  Only the parallel
 * port programmer is currently implemented.
 */
int
Gui::DoPort(Gui *me, Tcl_Interp *interp, int argc, char *argv[]) {
  const char *arg = argv[2];
  if(!arg[0]) {
    // Query for current port...
    Tcl_AppendResult(interp, me->cPic.SetPort(), 0);
  } else {
    arg = me->cPic.SetPort(arg);
    Tcl_AppendResult(interp, "Port now ", arg, "\n", 0);
  }
  return(0);
}

/* Read the PIC's memory into ram */
int Gui::DoRead(Gui *me, Tcl_Interp *interp, int argc, char *argv[]) {
    return(me->cPic.Read(CollectErrors, interp));
}

// Read pic memory into ram (all segments)
int Gui::DoErase(Gui *me, Tcl_Interp *interp, int argc, char *argv[]) {
  return(me->cPic.Erase(CollectErrors, interp));
}

// Verify ram image against PIC memory image
int Gui::DoVerify(Gui *me, Tcl_Interp *interp, int argc, char *argv[]) {
  return(me->cPic.Verify(CollectErrors, interp));
}

// Burn this pic...
int Gui::DoBurn(Gui *me, Tcl_Interp *interp, int argc, char *argv[]) {
  return(me->cPic.Write(CollectErrors, interp));
}

// Make list of what's available for chips...
int Gui::DoChips(Gui *me, Tcl_Interp *interp, int argc, char *argv[]) {
  const char **chips = me->cPic.Chips();
  for(int i = 0; chips[i]; i++) 
    // #### Blech!  Cast away const! Thank's alot Tcl!
    Tcl_AppendElement(interp, (char *)chips[i]);

  delete chips;
  return(0);
}

// Make list of pin names for the chosen programmer...
int Gui::DoPins(Gui *me, Tcl_Interp *interp, int argc, char *argv[]) {
  const char **pins = me->cPic.Pins();
  for(int i = 0; pins[i]; i++) 
    // #### Blech!  Cast away const! Thank's alot Tcl!
    Tcl_AppendElement(interp, (char *)pins[i]);
  delete pins;
  return(0);
}

// Get info on current chip's segments...
int Gui::DoInfo(Gui *me, Tcl_Interp *interp, int argc, char *argv[]) {
  const char *str = me->cPic.Info();
  // #### Blech!  Cast away const! Thank's alot Tcl!
  Tcl_SetResult(interp, (char *)str, TCL_VOLATILE);
  delete [] str;
  return(0);
}

// Get info on build version
int Gui::DoVersion(Gui *me, Tcl_Interp *interp, int argc, char *argv[]) {
  const char *str = me->cPic.Version();
  // #### Blech!  Cast away const! Thank's alot Tcl!
  Tcl_SetResult(interp, (char *)str, TCL_VOLATILE);
  delete [] str;
  return(0);
}

// Dump memory map info...  (could be big!)
// The C-core attempts to compress this, though...
int Gui::DoDump(Gui *me, Tcl_Interp *interp, int argc, char *argv[]) {
  ostrstream os;
  me->cPic.DumpRam(os);
  os << ends;
  char *str = os.str();
  Tcl_AppendResult(interp, str, 0);
  delete [] str;
  return(0);
}

// Instantiate a new connection object...
int Gui::TclNew(ClientData clientData, 
		Tcl_Interp *interp,
		int argc, char *argv[]) {
  if(argc != 2) {
    Tcl_AppendResult(interp, "Wrong number of arguments to ", argv[0], "\n", 0);
    return(TCL_ERROR);
  }
  char *name = argv[1];
  if(!Tcl_CreateCommand(interp, name, Gui::TclExec,
			(ClientData) new Gui(), Gui::TclDestroy)) {
    Tcl_AppendResult(interp, "Error creating '", name, "'\n", 0);
    return(TCL_ERROR);
  }

  Tcl_AppendResult(interp, name, 0);
  return TCL_OK;
}

// Destroy this object...
void
Gui::TclDestroy(ClientData clientData) {
  Gui *me = (Gui *)clientData;
  delete me;
}

// Register instantiation command for this object...
int
Gui::TclInit(Tcl_Interp *interp, char *cmd) {
  Tcl_CreateCommand(interp, cmd, TclNew,
		    (ClientData) NULL,
		    (Tcl_CmdDeleteProc *) NULL);
  return(TCL_OK);
}

/* AppInit(...) - initialize Tk application, add our (one) command
 */
int
Gui::AppInit(Tcl_Interp *interp) {

  for(;;) {			// Control construct only, never loops...

    if(Tcl_Init(interp) != TCL_OK) break;
    if(Tk_Init(interp) != TCL_OK) break;

    Tcl_StaticPackage(interp, "Tk", Tk_Init, Tk_SafeInit);

#ifdef HAVE_LIBITK3_0
    /* Do incrTk initinitialization... */
    if(Itcl_Init(interp) != TCL_OK) break;
    if(Itk_Init(interp) != TCL_OK) break;

    // What's all this?!  Needed, but stolen from itk distribution...
    Tcl_StaticPackage(interp, "Itcl", Itcl_Init, Itcl_SafeInit);
    Tcl_StaticPackage(interp, "Itk", Itk_Init, (Tcl_PackageInitProc *) NULL);

    if(Tcl_Import(interp, Tcl_GetGlobalNamespace(interp),
		  "::itk::*", /* allowOverwrite */ 1) != TCL_OK)  break;
    if(Tcl_Import(interp, Tcl_GetGlobalNamespace(interp),
		  "::itcl::*", /* allowOverwrite */ 1) != TCL_OK) break;
    if(Tcl_Eval(interp, "auto_mkindex_parser::slavehook { _%@namespace import -force ::itcl::* ::itk::* }") != TCL_OK) break;
#endif // HAVE_LIBITK3_0

    // Our specific extensions below here...
    if(TclInit(interp) != TCL_OK) break;

    // Whew!  Ran the gauntlet and made it...  
    return(TCL_OK);
  }
  // Here on any error...
  return(TCL_ERROR);
}

/* This callback is invoked by the core after every programmer operation
 * It can be used to report programming progress or error information
 * Programming errors are indicated when (want != got)
 * As long as this routine returns true, the operation is
 * allowed to continue.  If it returns false, the operation
 * is aborted.
 */
bool 
Gui::CollectErrors(void *obj, const char *seg, uint size,
		 uint addr, uint want, uint got) {
  Tcl_Interp *interp = (Tcl_Interp *) obj;
  char tmp[100];
  int abort;
  sprintf(tmp, "Progress {Segment: %s, Address: [0x%04x]}; update", 
	  seg, addr);
  Tcl_Eval(interp, tmp);

  // Quit?  User would have set abort...
  Tcl_GetInt(interp, Tcl_GetVar(interp, "abort", 0), &abort);
  if(want == got) return(!abort);

  sprintf(tmp, "Display {  [0x%04x] = 0x%04x != 0x%04x} append", 
	  addr, got, want);
  Tcl_Eval(interp, tmp);
  return(!abort);
}


// Wish top level...
int
main(int argc, char *argv[])
{
  Tk_Main(argc, argv, Gui::AppInit);
  return(0);
}

