/* Copyright (C) 1998, Mike Butler, mgb@mitre.org
 *    
 * This file is part of picptk, a PIC programmer for Linux
 *
 * Picptk is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * Picptk is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with picptk; see the file COPYING.  If not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * $Id: PicSerial.C,v 1.3 1998/12/31 01:40:33 mgb Exp $
 *
 * Serial port programmer driver...
 *
 * Serial port programmers use use the TX, RX, RTS, and DTR pins
 * to drive the PIC pins.  I think the mapping looks something 
 * like this:  (I think this is right...)
 *     Clk  ==> RTS
 *     Vpp  ==> TX data (via a kernel patch)
 *     Vdd  ==> ?
 *     Do   ==> DTR
 *     Din  ==> RX data
 *
 * I don't think serial programmers can control Vdd independently,
 * and they need a non-standard tty driver to hold TX low indefinately.
 * The driver appears to be a seperate kernel patch...
 */

#include <PicSerial.h>
#include <unistd.h>		// for open(), close(), ...

#include <termios.h>		// all these for serial port support
#include <sys/ioctl.h>
#include <fcntl.h>

// Serial port specific stuff below here...
/* Serial port programmer requires software control of
 * TxD, RTS, and DTR signals to work...
 * Unfortunately, these are all controlled in a non-standard
 * way with ioctl() calls.  Furthremore, the needed TxD control
 * ioctl isn't built into the kernel by default...
 */

// Open, init, etc...
SerialDriver::SerialDriver(Environment &env) : Driver(env) {
  
#ifndef TIOCCBRK
  throw("No TIOCCBRK ioctl?!
You probably can't use a serial port
programmer unless you install a kernel
patch.");

  // That'll get user's attention, now let compile it anyway...
#define TIOCCBRK 0
#endif
#ifndef TIOSCBRK
  throw("No TIOCCBRK ioctl?!
You probably can't use a serial port
programmer unless you install a kernel
patch.");
#define TIOSCBRK 0
#endif // TIOSCBRK

  const char *tty = cEnv.Get("port", "/dev/picp");

  if((cPort = open(tty, O_RDWR)) < 0)
    throw("Can't open serial port...");

  tcgetattr(cPort, &cState);

  struct termios term;
  term = cState;
  term.c_oflag = 0;
  term.c_lflag = 0;
  term.c_iflag = IGNBRK | IGNPAR;
  term.c_cflag = B9600 | CS8 | CREAD | CLOCAL;
  tcsetattr(cPort, TCSANOW, &term);
}

SerialDriver::~SerialDriver()
{
  ioctl(cPort, TIOCCBRK, 0);
  tcsetattr(cPort, TCSANOW, &cState);
  close(cPort);
}

// Set specified pin...
void SerialDriver::SetPin(Pic::Pin_t pin, int val = 1) {
  int tmp;
  val &= 1;
  switch(pin) {
  case Pic::pin_clk:
    tmp = TIOCM_RTS;
    ioctl(cPort, val ? TIOCMBIS : TIOCMBIC, &tmp);
    break;
  case Pic::pin_vpp:
    ioctl(cPort, val ? TIOSCBRK : TIOCCBRK, 0);
    break;
  case Pic::pin_vdd:
    // Is there a Vdd pin control?
    break;
  case Pic::pin_do:
    tmp = TIOCM_DTR;
    ioctl(cPort, val ? TIOCMBIS : TIOCMBIC, &tmp);
    break;
  default:
    throw("Can't write that serial port pin!");
  }
}

// Set specified pin...
bool SerialDriver::GetPin(Pic::Pin_t pin) {
  int val;
  switch(pin) {
  case Pic::pin_di:
    ioctl(cPort, TIOCMGET, &val);
    break;
  default:
    throw("Can't read that serial port pin!");
  }
  return(val);
}

// If serial port programmer pin bindings different among designs,
// then they need to be set as preferences.
// Change this if per-programmer pin assignemnts are necessary.
const char **SerialDriver::Pins() {
  return(0);
}
