/* Copyright (C) 1998, Mike Butler, mgb@mitre.org
 *    
 * This file is part of picptk, a PIC programmer for Linux
 *
 * Picptk is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * Picptk is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with picptk; see the file COPYING.  If not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * $Id: PicParallel.C,v 1.3 1998/12/31 01:40:32 mgb Exp $
 *
 * Programmer Drivers...
 * This is kind of intermediate glue, it captures the details
 * of specific programmers (what pins, what senses, ...)
 *
 * These routines map the Core's "Set Clk High" style of commands
 * into actions...  
 */

#include <PicParallel.h>
#include <unistd.h>		// for open(), close(), ...

// Set up table of pins we need to know about...
ParallelDriver::PinTbl_t ParallelDriver::cPinTbl[] = {
  {Pic::pin_vpp, "Vpp" ,  "5"},  
  {Pic::pin_vdd, "Vdd" ,  "4"},  
  {Pic::pin_clk, "Clk" ,  "~3"}, 
  {Pic::pin_do , "Dout",  "~2"}, 
  {Pic::pin_di , "Din" ,  "~10"},
};


// Parallel port specific stuff below here...
ParallelDriver::ParallelDriver(Environment &env) : Driver(env) {
  const char *tmp;

  /* Initilize port-pin to pic-pin mapping...
   * Do this before attempting to open the port...
   * If we're not root we'll still get the prefs defined...
   * This might be specific to each per-programmer design...
   * Defaults are based on my ITU PIC-1 programmer.
   */
  for(unsigned int i = 0; i < sizeof(cPinTbl) / sizeof(PinTbl_t); i++) {
    tmp = cEnv.Get(cPinTbl[i].name, cPinTbl[i].def);
    int pin = 0;
    if(sscanf(tmp, "~%d", &pin)) 
      pin = -pin;
    else if(sscanf(tmp, "!%d", &pin)) 
      pin = -pin;
    else sscanf(tmp, "%d", &pin);
    // there is no pin zero on the connector, so...
    if(!pin) {
      // Attribute's value is trashed!
      char s[100];
      sprintf(s, "Format error on configuration of pin '%s'", 
	      cPinTbl[i].name);
      throw(s);
    }
    cPinMap[cPinTbl[i].index] = pin;
  }

  // What port are we using?
  tmp = cEnv.Get("port", "lpt1");
  int pport;

  if(sscanf(tmp, "lpt%d", &pport)) {
    cPort = new ParallelPort(pport);
  } else if(sscanf(tmp, "LPT%d", &pport)) {
    cPort = new ParallelPort(pport);
  } else throw("Invalid printer port name");

}

// Set specified pin...
void ParallelDriver::SetPin(Pic::Pin_t pin, int val = 1) {
  int pinnum = cPinMap[pin];
  if(pinnum < 0) cPort->SetPin(-pinnum, val^1);
  else cPort->SetPin(pinnum, val);
}

// Get specified pin...
bool ParallelDriver::GetPin(Pic::Pin_t pin) {
  int pinnum = cPinMap[pin];
  if(pinnum < 0) return(cPort->GetPin(-pinnum) ^ 1);
  return(cPort->GetPin(pinnum));
}
// Pins() - Give caller a list of pin names
// Caller should free the resulting table (but not the elements)
const char **
ParallelDriver::Pins() {
  // Count number of chip entries...
  int count = sizeof(cPinTbl) / sizeof(PinTbl_t);

  const char **tbl = new const char*[count+1];
  for(int i = 0; i <= count; i++) {
    tbl[i] = cPinTbl[i].name;
  }
  return(tbl);
}
