/* Copyright (C) 1998, Mike Butler, mgb@mitre.org
 *    
 * This file is part of picptk, a PIC programmer for Linux
 *
 * Picptk is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * Picptk is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with picptk; see the file COPYING.  If not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 * 
 * $Id: PicMem.h,v 1.2 1998/12/31 01:40:32 mgb Exp $
 */
#ifndef Segment_h
#define Segment_h

#include <stdlib.h>		// for uint
#include <iostream.h>
#include <stdio.h>		// for sprintf()
#include <string.h>		// for strcat()


/* Abstraction of a single pic memory segment...
 * Contains details on how to write a specific type of memory,
 * size of memory, timing, etc...
 */
class Segment {
public:
  typedef enum { none,
		 eprom_conf,
		 eprom_prog,
		 flash_conf,
		 flash_data,
		 flash_prog,
  } Style_t;

  Style_t cStyle;		// type of memory (e.g., how to program) 
  char *cName;			// offset Segment name
  int cSkip;			// offset from base (in pic) 
  int cWords;			// length of segment 
  int cBase;			// offset from base of ram 
  int cBits;			// width of each word...
  bool cOmit;			// omit this segment for "each" processing
  uint *cRam;			// memory image for this portion
  char *cDirty;			// Map of used locations
private:

public:
  void Init() {
    delete [] cRam;
    cRam = new uint[cWords];
    cDirty = new char[cWords];
    for(int i = 0; i < cWords; i++) {
      cDirty[i] = 0;
      cRam[i] = (1<<cBits)-1;
    }
  }

  /* Damn!  This exposes a bunch of stuff to the programmer... */
  Style_t Style() const { return(cStyle); }
  int Base() const { return(cBase); }
  int Skip() const { return(cSkip); }
  int Length() const { return(cWords); }
  int Width() const { return(cBits); }
  bool Omit() const { return((cStyle == none) || cOmit); }
  const char *Name() const { return(cName); }

  // This to support byte writes for loading, etc...
  void SetByte(uint bAddr, unsigned char val) {
    int wAddr = bAddr >> 1;
    int offset = wAddr - cBase;
    if((offset < 0) || (offset >= cWords)) 
      throw ("Write to nonexistant PIC memory attempted");
    if(!cRam) Init();
    cDirty[offset] = 1;
    if(bAddr & 1) {
      cRam[offset] &= 0xff;
      cRam[offset] |= val << 8;
    } else {
      cRam[offset] &= ~0xff;
      cRam[offset] |= val;
    }
  }

  // Set value given word address...
  void SetWord(uint wAddr, uint val) {
    int offset = wAddr - cBase;
    if((offset < 0) || (offset >= cWords)) 
      throw ("Write to nonexistant PIC memory attempted");
    if(!cRam) Init();
    cRam[offset] = val & ((1<<cBits)-1);
    cDirty[offset] = 1;
  }

  // Get something at a word address...
  uint GetWord(uint wAddr) {
    int offset = wAddr - cBase;
    if((offset < 0) || (offset >= cWords)) 
      throw ("Read from nonexistant PIC memory attempted");
    if(!cRam) Init();
    return(cRam[offset] & (1<<cBits)-1);
  }

  // Determine if a word is "dirty"
  uint IsDirty(uint wAddr) {
    int offset = wAddr - cBase;
    if((offset < 0) || (offset >= cWords)) 
      throw ("Read from nonexistant PIC memory attempted");
    if(!cRam) Init();
    return(cDirty[offset]);
  }

  void Dump(ostream &os) {
    // Null bank?  Just leave...
    if(!cRam) return;
    if(Omit()) return;

    char line[200], data[200], prev[200];
    prev[0] = 0;
    uint count = 0;
    sprintf(line, "segment=%s, base=0x%04X, len=0x%04X", cName, cBase, cWords);
    os << line << endl;
    for(int addr = 0; addr < cWords; ) {
      sprintf(line, "  [0x%04X]=", cBase+addr);
      data[0] = 0;
      for(int i = 0; (i < 8) && (addr < cWords); i++, addr++) {
	char tmp[20];
	sprintf(tmp, "%04X ", cRam[addr]);
	strcat(data, tmp);
      }
      if(!strcmp(data, prev)) count++;
      else {
	if(count) {
	  os << "     repeats " << count << " more times" << endl;
	  count = 0;
	}
	os << line << data << endl;
	strcpy(prev, data);
      }
    }
    // Run right off end?  Say proper thing...
    if(count) os << "     repeats " << count << " more times" << endl;
  }
};

#endif // Segment
