/* Copyright (C) 1998, Mike Butler, mgb@mitre.org
 *    
 * This file is part of picptk, a PIC programmer for Linux
 *
 * Picptk is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * Picptk is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with picptk; see the file COPYING.  If not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 * 
 * $Id: PicCore.h,v 1.4 1998/12/31 01:40:32 mgb Exp $
 */
#ifndef Pic_h
#define Pic_h

#include <fstream.h>
#include <stdio.h>
#include <stdlib.h>
#include <UtEnvLite.h>		// for Environment
#include <PicMem.h>		// for Segment

class Driver;

// Description of a particular device's memory layout...
// This is a container class only; it allows an entire pic's memory
// layout to be described in a statically initialized array...
class MMap {
  friend class Pic;
private:
  enum { kMaxSeg = 10 };	// Maximum number of disjoint segments
public:
  char *cName;			// Device name
  char *cAlias;			// Use this spec instead...
  Segment cSegments[kMaxSeg];	// Memory segment descriptions
};


class Pic {
public:
  // Pins the driver should know about...
  // Since the driver is not derived, they must be exposed...
  enum Pin_t { 
    pin_vpp,
    pin_vdd,
    pin_clk,
    pin_do,
    pin_di,
    pin_last,			// sentinal, number of pins used...
  };

protected:
  static MMap cAllPics[];
  MMap *cMap;			// Memory map for selected pic
  Driver *cDriver;		// Parallel or serial port driver

  static void SetByte(void *obj, uint addr, char val);
  Environment cEnv;		// Preferences

  // Commands the PIC knows about... 
  // (CAUTION! PIC documents inconsistent with names!)
  enum Command { load_conf     = 0x00, 
		 unprotect0    = 0x01,
		 load_prog     = 0x02, 
		 load_data     = 0x03,
		 read_prog     = 0x04, 
		 read_data     = 0x05,
		 unprotect1    = 0x07,
		 erase_prog    = 0x09,
		 erase_data    = 0x0b,
		 increment     = 0x06,
		 begin_cycle   = 0x08,
		 end_cycle     = 0x0e,
				// Special flags below here...
		 clear_prot    = 0xfffe, // Illegal!
		 no_op	       = 0xffff, // Illegal!
  };

  // Details on timing and control for specific memory types...
  static struct Details_t {
    Segment::Style_t style;
    Command select;
    Command read;
    Command write;
    Command erase;
    Command end;
    int first;
    int over;
    int width;
  } cDetails[];
  // Look up details for a specific segment
  static Details_t *FindDetails(const Segment &seg);

  enum { kMaxTry = 25 };	// Try several on eprom writes

  // Things to mung with the PIC...
  // LSB firsts, any (reasonable) number of bits...
  virtual void SerialWrite(uint data, uint bits);
  virtual uint SerialRead(uint bits);
  virtual void Next() { SendCmd(increment); }
  virtual void PowerOn();
  virtual void PowerOff();
  virtual void SendCmd(uint cmd)   { SerialWrite(cmd, 6); }
  virtual void SendData(uint data) { SerialWrite(data<<1, 16); }
  virtual uint GetData(int bits = 14) { 
    uint val = SerialRead(16);	// Always read 16 bits?
    return((val >> 1) & ((1<<bits)-1));
  }

  // Call back used to report progress of long operations
  typedef bool (*ProgressCb_t)(void *obj, const char *seg, 
			       uint size, uint addr, 
			       uint want, uint got);
  // Type of operation that EachSegment mapcar can deal with...
  typedef int (Pic::*SegOpFunc_t)(Segment &, ProgressCb_t, void *);

  int EachSegment(SegOpFunc_t op, ProgressCb_t cb, void *obj, bool chip = true);

  // Then a bunch of one segment functions to feed it...
  int ReadSegment  (Segment &seg, ProgressCb_t cb=0, void *arg=0);
  int WriteSegment (Segment &seg, ProgressCb_t cb=0, void *arg=0);
  int VerifySegment(Segment &seg, ProgressCb_t cb=0, void *arg=0);
  int EraseSegment (Segment &seg, ProgressCb_t cb=0, void *arg=0);
  int InitSegment  (Segment &seg, ProgressCb_t cb=0, void *arg=0) { seg.Init(); return(0); }
  int DumpSegment  (Segment &seg, ProgressCb_t cb=0, void *arg=0) { seg.Dump((ostream &)*arg); return(0); }
  int BlankCheckSegment (Segment &seg, ProgressCb_t cb=0, void *arg=0);

  // Set/Get based on Pin name
  void SetPin(Pin_t pin, int val = 1);
  bool GetPin(Pin_t pin);

public:
  Pic();
  virtual ~Pic() {}

  // Muck with preferences...
  void ClearPrefs() { cEnv.Clear(); }
  bool LoadPrefs(const char *file = 0) { return(cEnv.Load(file)); }
  bool SavePrefs(const char *file = 0) { return(cEnv.Save(file)); }
  const char *SetPref(const char *attr, const char *val) { return(cEnv.Set(attr, val)); }
  const char *GetPref(const char *attr) { return(cEnv.Get(attr)); }

  // Make user functions apply to all segments...
  int Read  (ProgressCb_t cb = 0, void *obj = 0) { return(EachSegment(&ReadSegment,   cb, obj)); }
  int Write (ProgressCb_t cb = 0, void *obj = 0) { return(EachSegment(&WriteSegment,  cb, obj)); }
  int Verify(ProgressCb_t cb = 0, void *obj = 0) { return(EachSegment(&VerifySegment, cb, obj)); }
  int Erase (ProgressCb_t cb = 0, void *obj = 0) { return(EachSegment(&EraseSegment,  cb, obj)); }
  int DumpRam (ostream &os)                        { return(EachSegment(&DumpSegment,    0, &os, false)); }
  int LoadRam (const char *file);
  int InitRam ()                                 { return(EachSegment(&InitSegment,    0, 0, false)); }

  // Configuration functions...
  virtual const char **Chips();		// What chip types we support...
  virtual const char **Pins();		// List of pins (for this programmer)
  virtual const char **Segments();	// Segment IDs for current chip
  virtual const char *Info();		// Printable info about chip...
  virtual const char *Version();	// Version info...
  virtual const char *SetChip(const char *name = 0);
  virtual const char *SetPort(const char *port = 0);
  
};

#endif // Pic_h
