/* Copyright (C) 1998, Mike Butler, mgb@mitre.org
 *    
 * This file is part of picptk, a PIC programmer for Linux
 *
 * Picptk is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * Picptk is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with picptk; see the file COPYING.  If not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * $Id: PicCore.C,v 1.4 1998/12/31 01:40:32 mgb Exp $
 */
#include <PicCore.h>
#include <IoIntel.h>
#include <string.h>		// for strcat()

#include <PicParallel.h>	// Parallel port driver
#include <PicSerial.h>		// Serial port driver
#include <UtRealTime.h>
#include <unistd.h>		// for open(), close(), ...

#include <termios.h>		// all these for serial port support
#include <sys/ioctl.h>
#include <fcntl.h>
// cDetails[] - Table of PIC memory type details
/* Details on how to manipulate specific memory types...
 * Includes write pules with, iterations, overwrite, etc...
 * This is *not* where pic segments are defined... keep looking...
 */
Pic::Details_t Pic::cDetails[] = {
  //              style,    select,      read,     write, erase,           end, frst, ovr, width,
  { Segment::eprom_conf, load_conf, read_prog, load_prog, no_op,     end_cycle,   25,   3,   100},
  { Segment::eprom_prog,     no_op, read_prog, load_prog, no_op,     end_cycle,   25,   3,   100},
  { Segment::flash_conf, load_conf, read_prog, load_prog, clear_prot,    no_op,    0,   1, 10000},
  { Segment::flash_data,     no_op, read_data, load_data, erase_data,    no_op,    0,   1, 10000},
  { Segment::flash_prog,     no_op, read_prog, load_prog, erase_prog,    no_op,    0,   1, 10000},
  { Segment::none },
};
// SetChip() - Set the chip type we want to deal with
/* Set programming info from PIC name...
 * A call with a null arg is a query
 * (so the gui can reflect the current state without knowing the
 *  attribute name)
 */
const char *Pic::SetChip(const char *name) {
  if(!name) return(cEnv.Get("pic"));

  cMap = 0;
  int alias = 10, index = 0;
  while(alias && cAllPics[index].cName) {
    if(!strcmp(name, cAllPics[index].cName)) {
      if(cAllPics[index].cAlias) {
	name = cAllPics[index].cAlias;
	index = 0;
	alias--;			// Chase down alias...
	continue;
      } else {
	cMap = &cAllPics[index];	// Found bogie
	break;
      }
    }
    index++;
  }

  // Check if there was an error...
  if(!alias) throw("Pic name alias loop?!");
  if(!cMap)  throw("Unknown PIC type...");

  // Save choice as a preference...
  return(cEnv.Set("pic", name));
}
// Pic() - Constructor
// Initialize variables, read configuration, ...
Pic::Pic() {
  cMap = 0;			// Null memory map
  cDriver = 0;			// And no programmer driver yet...

  // Attempt to get configuraton information and preferences...
  char *home = getenv("HOME"), file[200];
  sprintf(file, "%s/%s", home ? home : ".", ".picprc");
  cEnv.DefaultFile(file);
  cEnv.Load();
}
// Load() - Initialize pic memory image from a file
/* Load file (only intel hex format?) into pic memory...
 * Thoughts:
 *    Could also "append" on read, is that desirable?
 */
int
Pic::LoadRam(const char *file) {
  if(!cMap) throw("Select a PIC type first");
  ifstream is(file);
  if(!is) throw("Can't open input hex file");

  InitRam();


  // Note... may throw an exception...
  return(IntelHex::Load(is, SetByte, this));
}
// SetByte() - Set a byte in pic memory (callback from IntelHex::Load)...
void
Pic::SetByte(void *obj, uint bAddr, char val) {
  Pic *me = (Pic*)obj;
  int match = 0;
  if((!me) || !me->cMap) return;		// No memory...
  for(uint i = 0; i < me->cMap->kMaxSeg; i++) {
    Segment &seg = me->cMap->cSegments[i];
    try {
      seg.SetByte(bAddr, val);
      match++;
    } catch (const char *e) { }
  }
  // Find mapping?  If not, this ain't right input!
  if(!match) throw ("File references nonexistant PIC memory?!");
}
// Chips() - Return list of PIC chips we support...
// Caller should free the resulting table (but not the elements)
const char **
Pic::Chips() {
  // Count number of chip entries...
  int count;
  for(count = 0; cAllPics[count].cName; count++) ;

  const char **tbl = new const char*[count+1];
  for(int i = 0; i <= count; i++) {
    tbl[i] = cAllPics[i].cName;
  }
  tbl[count] = 0;
  return(tbl);
}
// Pins() Ask programmer interface what pins it knows about...
// Pass this to the driver since it's dependent on him...
const char **
Pic::Pins() {
  if(!cDriver) throw ("Select a programmer port first");
  return(cDriver->Pins());
}
// Segments() return list of all segments in this chip
// Caller should free table of pointers (but not elements)
const char **
Pic::Segments() {
  if(!cMap) throw("Select a PIC type first");
  const char **tbl = new const char*[cMap->kMaxSeg+1];
  for(uint i = 0; i < cMap->kMaxSeg; i++) {
    tbl[i] = cMap->cSegments[i].Name();
  }
  tbl[cMap->kMaxSeg] = 0;
  return(tbl);
}
// Info() - Give printable version of segment info for current PIC...
// This is essentially a memory map but could include more...
const char *
Pic::Info() {
  if(!cMap) throw("Select a PIC type first");
  char *buf = new char[100];
  sprintf(buf, "Device: %s\n  Segments:\n   %13s %6s  %10s\n",
	  cMap->cName, "Address", "Width", "Name");
  for(int i = 0; cMap->cSegments[i].Name(); i++) {
    char line[100], *tmp = 0;
    Segment &seg = cMap->cSegments[i];
    sprintf(line, "   0x%04X-0x%04X %2d-bit  %10s\n",
	    seg.Base(), seg.Base() + seg.Length(), seg.Width(), seg.Name());
    tmp = new char[strlen(buf)+strlen(line)+1];
    strcpy(tmp, buf);
    strcat(tmp, line);
    delete [] buf;
    buf = tmp;
  }
  return(buf);
}

// Version() - Give version information...
// VERSION symbol is defined by configure script...
// Caller must delete allocated string...
const char *
Pic::Version() {
  char *tmp = new char[strlen(VERSION) + 1];
  strcpy(tmp, VERSION);
  return(tmp);
}
// EachSegment() - A form of mapcar for segments
// Perform a particular command for all memory segments...
int Pic::EachSegment(SegOpFunc_t func,
		     ProgressCb_t cb,
		     void *obj, bool needChip) {
  int errors = 0;
  if(!cMap) throw("Select a PIC type first");

  // This command need a programmer present?  (not all do)
  if(needChip) {
    if(!cDriver) throw("Select a programmer Port first");
    PowerOn();
  }

  for(uint i = 0; i < cMap->kMaxSeg; i++) {
    if(cMap->cSegments[i].Omit()) continue;
    errors += (this->*func)(cMap->cSegments[i], cb, obj);
  }

  // Turn power Vpp off (even if not on!)
  if(cDriver) PowerOff();

  return(errors);
}
// SetPort() - Choose the port we'll talk to...
// A null argument is a query for the current setting...
const char *Pic::SetPort(const char *port) {
  if(port) {
    /* Look at port and see if it's serial or parallel
     * If this kernel can't support a serial programmer,
     * then the constructor will throw an exception...
     * Terminals would begin with /dev/...  so...
     */
    if(port[0] == '/') cDriver = new SerialDriver(cEnv);
    else cDriver = new ParallelDriver(cEnv);

    cEnv.Set("port", port);
  }
  return(cEnv.Get("port"));
}
// FindDetails() - lookup programming details for particular memory type
Pic::Details_t *
Pic::FindDetails(const Segment &seg) {
  int i = 0;
  for(i = 0; cDetails[i].style != Segment::none; i++) {
    if(seg.Style() == cDetails[i].style) return(&cDetails[i]);
  }
  return(0);
}
// ReadSegment() - Read an entire segment from the PIC...
// For symmetry returns "errors," but always zero...
int
Pic::ReadSegment(Segment &seg, ProgressCb_t cb, void *arg) {
  int errors = 0;
  RealTime rt();		// Enter soft real-time block

  Details_t *dt = FindDetails(seg);
  if(!dt) throw("Unrecognized PIC memory type?");

  // Maybe select right segment...
  if(dt->select != no_op) {
    SendCmd(dt->select);
    SendData(0x00);		// dummy to get back to command state
  }

  // Skip initial portion of segment if needed...
  int skip = seg.Skip();
  while(skip--) SendCmd(increment);

  // Then do segment read proper...
  for(int i = 0; i < seg.Length(); i++) {
    SendCmd(dt->read);
    uint val = GetData(seg.Width());
    seg.SetWord(seg.Base() + i,  val);
    if(cb)
      if(!cb(arg, seg.Name(), seg.Length(), seg.Base() + i, val, val))
	return(errors + 1);	// User aborted!
    SendCmd(increment);
  }

  return(errors);		// always zero...
}
// VerifySegment() - Verify that PIC matches RAM image
// Verify that Pic memory matches ram image...
// returns error count...
int
Pic::VerifySegment(Segment &seg, ProgressCb_t cb, void *arg) {
  int errors = 0;

  Details_t *dt = FindDetails(seg);
  if(!dt) throw("Unrecognized PIC memory type?");

  // Maybe select right segment...
  if(dt->select != no_op) {
    SendCmd(dt->select);
    SendData(0x00);		// dummy to get back to command state
  }

  // Skip initial portion of segment if needed...
  int skip = seg.Skip();
  while(skip--) SendCmd(increment);

  // Verify only dirty locations?
  for(int i = 0; i < seg.Length(); i++) {
    if(!seg.IsDirty(seg.Base() + i)) {
      SendCmd(increment);
      continue;
    }
    int read, want;
    SendCmd(dt->read);
    read = GetData(seg.Width());
    want = seg.GetWord(seg.Base() + i);
    if(read != want)
      errors++;
    if(cb)
      if(!cb(arg, seg.Name(), seg.Length(), seg.Base() + i, want, read))
	return(errors + 1);	// User aborted!
    SendCmd(increment);
  }

  return(errors);
}
// EraseSegment() - Do electrical erase of PIC memory segment
// Special case for code protected configurations segments
int
Pic::EraseSegment(Segment &seg, ProgressCb_t cb, void *arg) {
  RealTime rt();		// Enter soft real-time block

  Details_t *dt = FindDetails(seg);
  if(!dt) throw("Unrecognized PIC memory type?");

  // If device is not eradable, do a blank check instead...
  if(dt->erase == no_op) 
    return(BlankCheckSegment(seg, cb, arg));

  if(dt->erase == clear_prot) {
    // Special case to turn off code protection and then erase...
    SendCmd(load_conf);
    SendData(0xffff);		// Code protect off into config word...
    for(int i = 0x2000; i < 0x2007; i++)
      SendCmd(increment);	// Skip to config word
    SendCmd(unprotect0);
    SendCmd(unprotect1);
    SendCmd(begin_cycle);
    RealTime::DelayUs(10000);
    SendCmd(unprotect0);
    SendCmd(unprotect1);
  } else {
    // Plain old single segment erase...
    SendCmd(load_data);
    SendData(0xffff);
    SendCmd(dt->erase);
    SendCmd(begin_cycle);
    RealTime::DelayUs(10000);
  }
  return(0);
}
// BlankCheckSegment() - See if this segment is blank...
int
Pic::BlankCheckSegment(Segment &seg, ProgressCb_t cb, void *arg) {
  RealTime rt();		// Enter soft real-time block

  Details_t *dt = FindDetails(seg);
  if(!dt) throw("Unrecognized PIC memory type?");

  // Maybe select right segment...
  if(dt->select != no_op) {
    SendCmd(dt->select);
    SendData(0x00);		// dummy to get back to command state
  }
  int skip = seg.Skip();
  while(skip--) SendCmd(increment);

  // Then do segment read proper...
  for(int i = 0; i < seg.Length(); i++) {
    SendCmd(dt->read);
    uint val = GetData(seg.Width());
    if(val ^ ((1 << seg.Width()) - 1)) {
      throw("Device is not blank and not electrically eraseable.");
    }
    SendCmd(increment);
    if(cb)
      if(!cb(arg, seg.Name(), seg.Length(), seg.Base() + i, val, val))
	return(1);		// User aborted!
  }
  return(0);
}

// WriteSegment() - Write one PIC memory segment
// Write entire segment to the chip...
// Returns error count...
int
Pic::WriteSegment(Segment &seg, ProgressCb_t cb, void *arg) {
  RealTime rt();		// Enter soft real-time block
  int errors = 0;		// Error count...

  Details_t *dt = FindDetails(seg);
  if(!dt) throw("Unrecognized PIC memory type?");

  // Get to proper segment...
  if(dt->select != no_op) {
    SendCmd(dt->select);
    SendData(0x00);
  }

  // Skip initial portion of segment if needed...
  int skip = seg.Skip();
  while(skip--) SendCmd(increment);

  // Then do segment write proper...  But only "dirty" locations
  for(int i = 0; i < seg.Length(); i++) {
    if(!seg.IsDirty(seg.Base() + i)) {
      SendCmd(increment);
      continue;
    }

    /* Generalized write procedure...
     * Pass in:
     *   first - number of tries for first success 0 --> no conditional
     *   over  - number of times to overburn (3X, 11X, ...)
     *   width - width of each programming pulse
     *   end   - true if "end_cycle" to be sent
     */
    uint tmp, val = seg.GetWord(seg.Base() + i);
    int tries;
    for(int pass = 0; pass < 2; pass++) {
      if(!pass) { tries = dt->first; }
      else {
	tries = (dt->first) ? (dt->over * (dt->first - tries)) : dt->over;
      }
      // Do programming cycles...
      for(; tries; tries--) {
	SendCmd(dt->write);
	SendData(val);
	SendCmd(begin_cycle);
	RealTime::DelayUs(dt->width);
	if(dt->end != no_op) SendCmd(dt->end);
	SendCmd(dt->read);
	tmp = GetData(seg.Width());
	if((tmp == val) && !pass) break;
      }
      if(dt->first && !tries) break;	// If write failed, early exit...
    }
    // Check on success, report errors...
    SendCmd(dt->read);
    tmp = GetData(seg.Width());
    if(tmp != val)
      errors++;
    if(cb)
      if(!cb(arg, seg.Name(), seg.Length(), seg.Base() + i, val, tmp))
	return(errors + 1);	// User aborted!
    SendCmd(increment);
  }

  return(errors);
}
// SerialWrite() - Serialize a byte or so for PIC...
// Send serial info to pic...  LSB first, any number of bits...
// Output bit is latched by receiver on falling edge...
// Note that some transfers include a start bit, stop bit, etc...
void
Pic::SerialWrite(uint data, uint bits) {
  if(bits > 8*sizeof(data)) throw("Too many bits to write");
  while(bits--) {
    SetPin(pin_clk, 1);
    SetPin(pin_do, data & 1);
    data >>= 1;
    SetPin(pin_clk, 0);
  }
  SetPin(pin_do, 1);	// data idles high...
  // Add a few microseconds of delay...
  SetPin(pin_do, 1);	// data idles high...
  SetPin(pin_do, 1);	// data idles high...
  SetPin(pin_do, 1);	// data idles high...
}
// SerialRead(...) - Deserialize a byte or so from PIC...
// Read specified number of bits from serial port...
// Data is latched on rising edge of clock...
// Note that some PIC transfers include a start bit, stop bit
// and other gibberish.  It's up to the caller to take care of
// all that!
uint
Pic::SerialRead(uint bits) {
  uint result = 0;
  if(bits > 8*sizeof(result)) throw("Too many bits to read");
  bool value;
  for(uint mask = 1; bits; bits--) {
    SetPin(pin_clk, 1);
    SetPin(pin_clk, 0);
    value = GetPin(pin_di);
    result |= (value) ? mask : 0;
    mask <<= 1;
  }
  return(result);
}
// PowerOn() PowerOff() routines init programmer...
// Turn on programming voltage...
void
Pic::PowerOn() {
  RealTime rt();		// Enter soft real-time block
  SetPin(pin_do, 0);
  SetPin(pin_clk, 0);
  SetPin(pin_vdd, 1);
  RealTime::DelayUs(10);	// Must allow Vdd to settle for 'C76?!
  SetPin(pin_vpp, 1);
}

// Turn off programming voltage...
void
Pic::PowerOff() {
  SetPin(pin_vpp, 0);
  SetPin(pin_vdd, 0);
  SetPin(pin_clk, 0);
  SetPin(pin_do, 0);
}
 // SetPin() - Hook to the driver's set pin...
void Pic::SetPin(Pin_t pin, int val = 1) {
  if(!cDriver) throw (cDriver);	// Get the hint?
  cDriver->SetPin(pin, val);
}

// GetPin() - Hook to the driver's function...
bool Pic::GetPin(Pin_t pin) {
  if(!cDriver) throw (cDriver);	// Bad driver, bad, bad, bad!
  return(cDriver->GetPin(pin));
}

// #### Debugging code below this point...
#if 0
// This callback is called after every programmer operation
// It can be used to report programming progress or error information
// Errors are indicated when (want != got)
static void Stat(void *obj, const char *seg, uint size,
		 uint addr, uint want, uint got) {
  char tmp[100];
  if((want == got) && (addr&0xff)) return;
  sprintf(tmp, "%c %s[0x%04x] = 0x%04x (0x%04x)",
	  (want == got) ? ' ' : '*', seg, addr, got, want);
  cout << tmp << endl;
}

main()
{
  try {
    Pic p;
    ParProgrammer ppp(1);
    p.SetChip("teste");
    p.SetProgrammer(ppp);
    p.Read(Stat);
    p.Dump(cout);
    p.Load("test.hex");
    p.Write(Stat);
    //p.Verify(Stat);
    //p.Dump(cout);
    //p.Read(Stat);
  } catch (const char *e) {
    cout << "Aborted: " << e << endl;
  }

}
#endif
