/* Copyright (C) 1998, Mike Butler, mgb@mitre.org
 *    
 * This file is part of picptk, a PIC programmer for Linux
 *
 * Picptk is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * Picptk is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with picptk; see the file COPYING.  If not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * $Id: IoIntel.C,v 1.2 1998/12/31 01:40:32 mgb Exp $
 *
 * Intel Hex file I/O formatting and such...
 *
 */
#include <IoIntel.h>
#include <stdio.h>		// for sscanf
// ParseLine(...) - parse one line of Intel Hex8 format
// Provide a call back that writes memory in byte-sized chunks.
// For the PIC, the four byte file string "FACE" is 
// gets packed into a 12-bit PIC word as 0xCEAF...
int
IntelHex::ParseLine(const char *line, CbSetByte set, void *obj) {
  uint len;
  int  addr, type, sum, used;
  
  // Format check...
  if(3 != sscanf(line, ":%02X%04X%02X%n",
		 &len, &addr, &type, &used)) 
    throw("Bad IntelHex record format");
  
  // Only know about "data" type...  Throw format errors!
  if(type == 0x01) return(0);
  if(type || !len) 
    throw("Unsupported IntelHex record type");
  
  const char *s = &line[used];
  sum = len + addr + type + (addr >> 8);
  if(2*(len+1) != strlen(s)) 
    throw("IntelHex record truncated!?");
  
  // Set value in segment memory (wherever it is...)
  for(uint byte = 0; byte <= len; byte++) {
    int val;
    if(1 != sscanf(&s[2*byte], "%02X", &val)) 
      throw("Bad IntelHex record format");
    sum += val;
    if(byte < len) set(obj, addr++, val);
  }
  
  if(sum & 0xff) throw("Bad checksum on IntelHex record");
  return(len);
}
// OutputLine();
// Write one line of data to output stream
// If length is zero, writes end of file marker...
// This should be smarter about supressing zeros, maybe?
bool
IntelHex::OutputLine(ostream &os, int len, uint addr = 0, char *buffer = 0) {
  char line[200];
  int sum = 0;
  int type = (len) ? 0 : 1;
  
  sum = len + addr + (addr >> 8);

  sprintf(line, ":%02X%04X%02X", len, addr, type);
  os << line;
  for(int i = 0; i < len; i++) {
    sum += buffer[i];
    sprintf(line, "%02X", buffer[i] & 0xff);
    os << line;
  }
  sprintf(line, "%02X", 0xff & ~sum);
  os << line << endl;
  return(true);
}



// Load(...) - read memory values from a file
// Read byte stream in Intel Hex8 fromat from file...
// Caller must provide write call back...
int
IntelHex::Load(istream &is, CbSetByte set, void *obj) {
  const int max = 300;
  int bytes = 0, tmp;
  char line[max];
  while(is) {
    is.getline(line, max);
    bytes += (tmp = ParseLine(line, set, obj));
    if(!tmp) return(bytes);	// Normal eof; everything ok...
  }
  throw ("Premature EOF encountered on input");
}
// Save(...) - save memory values from to a file
// There's a bit of a nasty asymmetry here...
// Reading must be able to place data in any memory location,
// Writing always wants to write lines in blocks...
// Therefore, SetByte and GetByte callbacks are fundamentally different.
bool
IntelHex::Save(ostream &os, CbGetByte get, void *obj) {
  uint addr = 0, base = addr, next = 0, len = 0;
  char byte, buffer[16];
  while(get(obj, addr, next, byte)) {
    if(addr+1 != next || (len == 16)) {
      if(len) OutputLine(os, len, base, buffer);
      base = next;
      len = 0;
    }
    buffer[len++] = byte;
    addr = next;
  }
  if(len) OutputLine(os, len, base, buffer);
  OutputLine(os, 0);		// end of file...
  return(true);
}
// main(...) - debugging code...
#if 0
const int length = 100;
static char memory[length];

bool 
Set(void *v, uint addr, char val) {
  if(addr < 0) return(false);
  if(addr >= length) return(false);
  memory[addr] = val;  
  return(true);
}
bool 
Get(void *v, uint &addr, uint &next, char &val) {
  if(addr >= length) return(false);
  next = addr+1;
  val = memory[addr];
  return(true);
}

main()
{
  ifstream is("test.hex");
  IntelHex::Load(is, Set, 0);
  IntelHex::Save(cout, Get, 0);
}

#endif
