/* Copyright (C) 1998, Mike Butler, mgb@mitre.org
 *    
 * This file is part of picptk, a PIC programmer for Linux
 *
 * Picptk is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * Picptk is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with picptk; see the file COPYING.  If not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * $Id: IfParPort.C,v 1.2 1998/12/31 01:40:31 mgb Exp $
 *
 * Abstraction of the parallel port...
 * Essentialy, this provides a Set and Get operation for
 * specific pins.
 *
 * ToDo:
 *    Byte wide transfers GetByte()
 *    Support bidiriectional pins 
 */
#include <IfParPort.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/io.h>		// for ioperm()


// Gain access to parallel port, return 0 on success...
bool 
ParallelPort::Access(uint port, bool allow) {
  int result;
  // Do setuid here to gain access?
  if(ioperm(port, kPortSize, allow)) 
    throw("Can't open printer port, try running as root.");
  // Then unsetuid...
  return(result);
}

ParallelPort::ParallelPort(uint port = 0) {

  // These are standard enough to not be in the environment...
  int ports[kMaxPorts] = { 0x3bc, 0x378, 0x278, };
  
  // Translation table courtesy of whom ?? ################
  PinInfo_t pinfo[kMaxPins] = {
    {  nc, 0, 1, 0x00 },	/* pin 0	Invalid pin #	*/
    { out, 1, 2, 0x01 },	/* pin 1	(out)	!strobe	*/
    { out, 0, 0, 0x01 },	/* pin 2	(out)	Data 0	*/
    { out, 0, 0, 0x02 },	/* pin 3	(out)	Data 1	*/
    { out, 0, 0, 0x04 },	/* pin 4	(out)	Data 2	*/
    { out, 0, 0, 0x08 },	/* pin 5	(out)	Data 3	*/
    { out, 0, 0, 0x10 },	/* pin 6	(out)	Data 4	*/
    { out, 0, 0, 0x20 },	/* pin 7	(out)	Data 5	*/
    { out, 0, 0, 0x40 },	/* pin 8	(out)	Data 6	*/
    { out, 0, 0, 0x80 },	/* pin 9	(out)	Data 7	*/
    {  in, 0, 1, 0x40 },	/* pin 10	(in)	!ack	*/
    {  in, 1, 1, 0x80 },	/* pin 11	(in)	busy	*/
    {  in, 0, 1, 0x20 },	/* pin 12	(in)	Pout	*/
    {  in, 0, 1, 0x10 },	/* pin 13	(in)	Select	*/
    { out, 1, 2, 0x02 },	/* pin 14	(out)	!feed	*/
    {  in, 0, 1, 0x08 },	/* pin 15	(in)	!error	*/
    { out, 0, 2, 0x04 },	/* pin 16	(out)	!init	*/
    { out, 1, 2, 0x08 },	/* pin 17	(out)	!SI	*/
    {  nc, 0, 1, 0x00 },	/* pin 18	GND		*/
  };
  cPinInfo = pinfo;
  if(port >= kMaxPorts) 
    throw("Bad printer port number");
  cPort = ports[port];
  Access(cPort, true);
  for(uint i = 0; i < kPortSize; i++) cState[i] = 0;
}

ParallelPort::~ParallelPort() {
  Access(cPort, false);
}

void
ParallelPort::SetPin(uint pin, bool val) {
  if(pin >= kMaxPins) 
    throw("Pin number out of range");
  const PinInfo_t &pi = cPinInfo[pin];
  if(pi.dir != out) throw("Output to input pin?");
  
  // Set or clear bit appropriately
  if(val ^ pi.sense) cState[pi.offset] |= pi.mask;
  else cState[pi.offset] &= ~pi.mask;
  
  // Then set bit...
  outb(cState[pi.offset], cPort+pi.offset);
}
// 
bool 
ParallelPort::GetPin(uint pin) {
  if(pin >= kMaxPins) throw("Pin number out of range");
  const PinInfo_t &pi = cPinInfo[pin];
  if(pi.dir != in) throw("Input from output pin?");
  
  // What happens if port has input and output pins?
  int tmp = inb(cPort+pi.offset);
  return(tmp & pi.mask);
}

// Byte wide write to data port...
// Should insure correct directional signals are sent?
void 
ParallelPort::WriteByte(char data) {
  outb(data, cPort);
}

// Byte wide read (should set data bus direction bits)
char
ParallelPort::ReadByte() {
  // inb(BASEPORT+1);
  return(0);
}

// main(...)  - debugging routine
#if 0
#include <iostream.h>
main()
{
  try {
    ParallelPort p(1);
    for(;;) {
      p.SetPin(1);
      sleep(1);
      p.ClrPin(1);
      sleep(2);
      cout << p.GetPin(10) << endl;
    }
    p.WriteByte(0);
    p.ReadByte();
  } catch (const char *e) {
    cerr << "Port problem, error: " << e << endl;
  }
}

#endif
